#include <linux/init.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/sched/signal.h>

static int pg_stats_show(struct seq_file *m, void *v) {
    struct task_struct *t;

    // use for_each_process so that it displays the stats for each process
    for_each_process(t) {
        seq_printf(m, "%d: [%lu,%lu,%lu], [%lu,%lu,%lu], [%lu,%lu,%lu], [%lu,%lu,%lu]\n",
            t->pid, 
            t->pgd_alloc_count, t->pgd_free_count, t->pgd_set_count, 
            t->pud_alloc_count, t->pud_free_count, t->pud_set_count,
            t->pmd_alloc_count, t->pmd_free_count, t->pmd_set_count, 
            t->pte_alloc_count, t->pte_free_count, t->pte_set_count);
    }
    return 0;
}

// wrapper for single_open()
static int pg_stats_open(struct inode *inode, struct file *file) {
    return single_open(file, pg_stats_show, NULL);
}

static const struct proc_ops pg_stats_ops = {
    // runs when someone opens /proc/pg_stats
    .proc_open    = pg_stats_open,
    // handles the reading of the file contents
    .proc_read    = seq_read,
    // for changing the read position inside a file
    .proc_lseek   = seq_lseek,
    // called after file is closed
    .proc_release = single_release,
};

static int __init pg_stats_init(void) {
    proc_create("pg_stats", 0, NULL, &pg_stats_ops);
    return 0;
}
static void __exit pg_stats_exit(void) {
    remove_proc_entry("pg_stats", NULL);
}
// needed since pg_stats.c is a self-contained kernel module
module_init(pg_stats_init);
module_exit(pg_stats_exit);
