#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched/signal.h>
#include <linux/atomic.h>

static void print_task_pg_stats(struct seq_file *m, struct task_struct *task) {
    seq_printf(m, "%d: ", task->pid);

    seq_printf(m, "[[%ld],[%ld],[%ld]], ", 
        atomic_long_read(&task->pgd_alloc), 
        atomic_long_read(&task->pgd_free), 
        atomic_long_read(&task->pgd_set));

    seq_printf(m, "[[%ld],[%ld],[%ld]], ", 
        atomic_long_read(&task->pud_alloc), 
        atomic_long_read(&task->pud_free), 
        atomic_long_read(&task->pud_set));

    seq_printf(m, "[[%ld],[%ld],[%ld]], ", 
        atomic_long_read(&task->pmd_alloc), 
        atomic_long_read(&task->pmd_free), 
        atomic_long_read(&task->pmd_set));

    seq_printf(m, "[[%ld],[%ld],[%ld]]\n", 
        atomic_long_read(&task->pte_alloc), 
        atomic_long_read(&task->pte_free), 
        atomic_long_read(&task->pte_set));
}

static int pg_stats_show(struct seq_file *m, void *v) {
    struct task_struct *task;

    rcu_read_lock();
    for_each_process(task) {
        print_task_pg_stats(m, task);
    }
    rcu_read_unlock();

    return 0;
}

static int pg_stats_open(struct inode *inode, struct file *file) {
    return single_open(file, pg_stats_show, NULL);
}

static const struct proc_ops pg_stats_fops = {
    .proc_open = pg_stats_open,
    .proc_read = seq_read,
    .proc_lseek = seq_lseek,
    .proc_release = single_release,
};

static int __init pg_stats_init(void) {
    proc_create("pg_stats", 0444, NULL, &pg_stats_fops);
    return 0;
}

static void __exit pg_stats_exit(void) {
    remove_proc_entry("pg_stats", NULL);
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);
MODULE_LICENSE("GPL");
