#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/sched/signal.h>
#include <linux/init.h>

// Function to display process memory statistics in /proc/pg_stats
static int pg_stats_show(struct seq_file *m, void *v) {
    struct task_struct *task;

    rcu_read_lock();
    for_each_process(task) {
        seq_printf(m, "%d: [%lu,%lu,%lu], [%lu,%lu,%lu], [%lu,%lu,%lu], [%lu,%lu,%lu]\n",
                   task->pid,
                   task->pgd_alloc, task->pgd_free, task->pgd_set,
                   task->pud_alloc, task->pud_free, task->pud_set,
                   task->pmd_alloc, task->pmd_free, task->pmd_set,
                   task->pte_alloc, task->pte_free, task->pte_set);
    }
    rcu_read_unlock();
    return 0;
}


// Open function for /proc/pg_stats
static int pg_stats_open(struct inode *inode, struct file *file) {
    return single_open(file, pg_stats_show, NULL);
}

// Use `proc_ops` instead of `file_operations`
static const struct proc_ops pg_stats_pops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

// Initialize /proc/pg_stats entry in the kernel
static int __init pg_stats_init(void) {
    struct proc_dir_entry *entry;

    entry = proc_create("pg_stats", 0444, NULL, &pg_stats_pops);
    if (!entry) {
        pr_err("Failed to create /proc/pg_stats\n");
        return -ENOMEM;
    }

    pr_info("/proc/pg_stats created successfully\n");
    return 0;
}

// Remove /proc/pg_stats when the module is unloaded
static void __exit pg_stats_exit(void) {
    remove_proc_entry("pg_stats", NULL);
    pr_info("/proc/pg_stats removed\n");
}

// Register module initialization and exit functions
module_init(pg_stats_init);
module_exit(pg_stats_exit);

