#include <linux/init.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/sched/signal.h>
#include <linux/rcupdate.h>  

/* Show function for /proc/pg_stats */
static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;

    rcu_read_lock();
    for_each_process(task) {
        /*
         * Print 12 counters in the format:
         *   [pid]: [[pgd_alloc,pgd_free,pgd_set],
         *           [pud_alloc,pud_free,pud_set],
         *           [pmd_alloc,pmd_free,pmd_set],
         *           [pte_alloc,pte_free,pte_set]]
         */
        seq_printf(m,
                   "[%d]: [[%lu,%lu,%lu], [%lu,%lu,%lu], [%lu,%lu,%lu], [%lu,%lu,%lu]]\n",
                   task->pid,
                   /* PGD */
                   task->pgd_alloc, task->pgd_free, task->pgd_set,
                   /* PUD */
                   task->pud_alloc, task->pud_free, task->pud_set,
                   /* PMD */
                   task->pmd_alloc, task->pmd_free, task->pmd_set,
                   /* PTE */
                   task->pte_alloc, task->pte_free, task->pte_set);
    }
    rcu_read_unlock();

    return 0;
}

/* Open function for /proc/pg_stats */
static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

/* File operations for /proc/pg_stats */
static const struct proc_ops pg_stats_fops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

/* Module init: create /proc/pg_stats */
static int __init pg_stats_init(void)
{
    proc_create("pg_stats", 0, NULL, &pg_stats_fops);
    pr_info("pg_stats: /proc/pg_stats created\n");
    return 0;
}

/* Module exit: remove /proc/pg_stats */
static void __exit pg_stats_exit(void)
{
    remove_proc_entry("pg_stats", NULL);
    pr_info("pg_stats: /proc/pg_stats removed\n");
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);

