#include <linux/module.h>
#include <linux/init.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/sched/signal.h>  /* for for_each_process */

static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;

    for_each_process(task) {
        seq_printf(m,
                   "[%d]: [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]], "
                   "[[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]]\n",
                   task->pid,
                   task->pgd_alloc_count, task->pgd_free_count, task->pgd_set_count,
                   task->pud_alloc_count, task->pud_free_count, task->pud_set_count,
                   task->pmd_alloc_count, task->pmd_free_count, task->pmd_set_count,
                   task->pte_alloc_count, task->pte_free_count, task->pte_set_count);
    }
    return 0;
}

static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

static const struct proc_ops pg_stats_ops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

static int __init pg_stats_init(void)
{
    struct proc_dir_entry *entry;

    entry = proc_create("pg_stats", 0, NULL, &pg_stats_ops);
    if (!entry) {
        pr_err("pg_stats: Failed to create /proc/pg_stats\n");
        return -ENOMEM;
    }
    pr_info("pg_stats: Module loaded, /proc/pg_stats created\n");
    return 0;
}

static void __exit pg_stats_exit(void)
{
    remove_proc_entry("pg_stats", NULL);
    pr_info("pg_stats: Module unloaded, /proc/pg_stats removed\n");
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);

