// fs/proc/pg_stats.c

#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/sched/signal.h>  // for_each_process() macro
#include <linux/init.h>

static struct proc_dir_entry *pg_stats_entry;

// The show function that prints the counters for each active process
static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;
        
    rcu_read_lock();
    // Iterate over every process in the system
    for_each_process(task) {
        seq_printf(m, "%d: "
            "[%lld,%lld,%lld], [%lld,%lld,%lld], [%lld,%lld,%lld], [%lld,%lld,%lld]\n",
            task->pid,

            // PGD
            atomic64_read(&task->pgd_alloc_count),
            atomic64_read(&task->pgd_free_count),
            atomic64_read(&task->pgd_set_count),

            // PUD
            atomic64_read(&task->pud_alloc_count),
            atomic64_read(&task->pud_free_count),
            atomic64_read(&task->pud_set_count),

            // PMD
            atomic64_read(&task->pmd_alloc_count),
            atomic64_read(&task->pmd_free_count),
            atomic64_read(&task->pmd_set_count),

            // PTE
            atomic64_read(&task->pte_alloc_count),
            atomic64_read(&task->pte_free_count),
            atomic64_read(&task->pte_set_count)
        );
    }
        
    rcu_read_unlock();
    return 0;
}

// The open function (seq_file boilerplate)
static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

static const struct proc_ops pg_stats_fops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek  = seq_lseek,
    .proc_release = single_release,
};

// Module init
static int __init pg_stats_init(void)
{
    pg_stats_entry = proc_create("pg_stats", 0, NULL, &pg_stats_fops);
    return 0;
}

// Module exit
static void __exit pg_stats_exit(void)
{
    remove_proc_entry("pg_stats", NULL);
    pr_info("pg_stats module unloaded.\n");
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);

