#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched/signal.h>

static int pg_stats_show(struct seq_file *m, void *v);
static int pg_stats_open(struct inode *inode, struct file *file);
static int __init pg_stats_init(void);
static void __exit pg_stats_exit(void);

// Define pg_stats_fops here
static const struct proc_ops pg_stats_fops = {
    .proc_open = pg_stats_open,
    .proc_read = seq_read,
    .proc_lseek = seq_lseek,
    .proc_release = single_release,
};

static int pg_stats_show(struct seq_file *m, void *v) {
    struct task_struct *task;

    // Header (optional)
    seq_puts(m, "Memory Management Statistics by PID:\n");

    // For each process, print all 12 values
    for_each_process(task) {
        seq_printf(m, "%d: [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]]\n",
            task->pid,
            // PGD: alloc, free, set
            atomic_long_read(&task->pgd_alloc),
            atomic_long_read(&task->pgd_free),
            atomic_long_read(&task->pgd_set),
            // PUD: alloc, free, set
            atomic_long_read(&task->pud_alloc),
            atomic_long_read(&task->pud_free),
            atomic_long_read(&task->pud_set),
            // PMD: alloc, free, set
            atomic_long_read(&task->pmd_alloc),
            atomic_long_read(&task->pmd_free),
            atomic_long_read(&task->pmd_set),
            // PTE: alloc, free, set
            atomic_long_read(&task->pte_alloc),
            atomic_long_read(&task->pte_free),
            atomic_long_read(&task->pte_set));
    }
    return 0;
}

static int pg_stats_open(struct inode *inode, struct file *file) {
    return single_open(file, pg_stats_show, NULL);
}

static int __init pg_stats_init(void) {
	pr_info("pg_stats: Initializing module...\n");
    struct proc_dir_entry *entry;

    entry = proc_create("pg_stats", 0444, NULL, &pg_stats_fops);
    if (!entry) {
        pr_err("Failed to create /proc/pg_stats\n");
        return -ENOMEM;
    }

    pr_info("Module pg_stats loaded: /proc/pg_stats created\n");
    return 0;
}

static void __exit pg_stats_exit(void) {
    remove_proc_entry("pg_stats", NULL);
    pr_info("Module pg_stats removed: /proc/pg_stats deleted\n");
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("William Baldwin");
MODULE_DESCRIPTION("Custom /proc/pg_stats for tracking memory statistics.");
