#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/init_task.h>
#include <linux/uaccess.h>
#include <linux/atomic.h>

#define PROC_NAME "pg_stats"  // Define the filename under /proc

/* 
 * Function to display the contents of /proc/pg_stats 
 * This function iterates through all processes and prints page table operation statistics.
 */
static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;
    // Print the header format: Process ID + page table operation statistics
    seq_printf(m, "pid: [[pgd_alloc],[pgd_free],[pgd_set]], "
                   "[[pud_alloc],[pud_free],[pud_set]], "
                   "[[pmd_alloc],[pmd_free],[pmd_set]], "
                   "[[pte_alloc],[pte_free],[pte_set]]\n");

    rcu_read_lock(); // Enter RCU read-side critical section to prevent concurrent modifications
    for_each_process(task) { // Iterate through all processes
        seq_printf(m, "%d: [[%d],[%d],[%d]], "
                       "[[%d],[%d],[%d]], "
                       "[[%d],[%d],[%d]], "
                       "[[%d],[%d],[%d]]\n",
                   task->pid, // Process ID
                   // Read PGD (Page Global Directory) statistics
                   atomic_read(&task->group_leader->pt_stats.pgd_alloc),
                   atomic_read(&task->group_leader->pt_stats.pgd_free),
                   atomic_read(&task->group_leader->pt_stats.pgd_set),

                   // Read PUD (Page Upper Directory) statistics
                   atomic_read(&task->group_leader->pt_stats.pud_alloc),
                   atomic_read(&task->group_leader->pt_stats.pud_free),
                   atomic_read(&task->group_leader->pt_stats.pud_set),

                   // Read PMD (Page Middle Directory) statistics
                   atomic_read(&task->group_leader->pt_stats.pmd_alloc),
                   atomic_read(&task->group_leader->pt_stats.pmd_free),
                   atomic_read(&task->group_leader->pt_stats.pmd_set),

                   // Read PTE (Page Table Entry) statistics
                   atomic_read(&task->group_leader->pt_stats.pte_alloc),
                   atomic_read(&task->group_leader->pt_stats.pte_free),
                   atomic_read(&task->group_leader->pt_stats.pte_set));
    }
    rcu_read_unlock();  // Exit RCU read-side critical section

    return 0;
}

/* 
 * Function to open /proc/pg_stats 
 * Uses the single_open() function to handle reading the statistics
 */
static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

/* 
 * Define file operations for /proc/pg_stats
 * These specify how the file is opened, read, and released.
 */
static const struct proc_ops pg_stats_ops = {
    .proc_open    = pg_stats_open,   // Function to handle opening the file
    .proc_read    = seq_read,        // Function to read data from the file
    .proc_lseek   = seq_lseek,       // Function to handle seeking in the file
    .proc_release = single_release,  // Function to release the file when closed
};

/* 
 * Module initialization function 
 * Creates the /proc/pg_stats file and registers its operations
 */
static int __init pg_stats_init(void)
{
    proc_create(PROC_NAME, 0, NULL, &pg_stats_ops);  // Create the /proc entry
    printk(KERN_INFO "pg_stats module loaded\n");    // Log module loading
    return 0;
}


/* 
 * Module cleanup function 
 * Removes the /proc/pg_stats entry when the module is unloaded
 */
static void __exit pg_stats_exit(void)
{
    remove_proc_entry(PROC_NAME, NULL);  // Remove the /proc entry
    printk(KERN_INFO "pg_stats module unloaded\n");  // Log module unloading
}

module_init(pg_stats_init);  // Register module initialization function
module_exit(pg_stats_exit);  // Register module exit function

MODULE_LICENSE("GPL");  // Set module license
MODULE_AUTHOR("Name");  // Set module author
MODULE_DESCRIPTION("Page Table Operation Statistics Module");  // Module description
