// SPDX-License-Identifier: GPL-2.0
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <linux/sched/signal.h> 
#include <linux/seq_file.h>

static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;

    // Iterate through all processes
    for_each_process(task) {
        seq_printf(m, "[%d]: [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]]\n",
                   task->pid,
                   task->pgd_alloc_count, task->pgd_free_count, task->pgd_set_count,
                   task->pud_alloc_count, task->pud_free_count, task->pud_set_count,
                   task->pmd_alloc_count, task->pmd_free_count, task->pmd_set_count,
                   task->pte_alloc_count, task->pte_free_count, task->pte_set_count);
    }

    return 0;
}

static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

static const struct proc_ops pg_stats_proc_ops = {
    .proc_flags    = PROC_ENTRY_PERMANENT,
    .proc_open     = pg_stats_open,
    .proc_read_iter = seq_read_iter,
    .proc_lseek    = seq_lseek,
    .proc_release  = single_release,
};

static int __init proc_pg_stats_init(void)
{
    // Create the /proc/pg_stats entry
    proc_create("pg_stats", 0, NULL, &pg_stats_proc_ops);
    pr_info("pg_stats module loaded\n");
    return 0;
}

static void __exit proc_pg_stats_exit(void)
{
    // Remove the /proc/pg_stats entry
    remove_proc_entry("pg_stats", NULL);
    pr_info("pg_stats module unloaded\n");
}

module_init(proc_pg_stats_init);
module_exit(proc_pg_stats_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Page Table Statistics Module");
MODULE_AUTHOR("Oscar Xu");
