#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched/signal.h>
#include <linux/module.h>

#define PROC_FILENAME "pg_stats"

/**
 * pg_stats_show - Function to display process memory management statistics
 * @m: Pointer to the seq_file structure (used for formatted output)
 * @v: Unused argument (required by seq_file API)
 *
 * This function iterates over all processes and prints their page table operation
 * counters in a structured format. The output is structured as:
 *
 * [PID]: [[PGD_alloc, PGD_free, PGD_set]], [[PUD_alloc, PUD_free, PUD_set]],
 *        [[PMD_alloc, PMD_free, PMD_set]], [[PTE_alloc, PTE_free, PTE_set]]
 */

static int pg_stats_show(struct seq_file *m, void *v) {
    struct task_struct *task;
	
    // Iterate through all tasks (processes) in the system
    for_each_process(task) {
        seq_printf(m, "%d: [[%lu, %lu, %lu]], [[%lu, %lu, %lu]], [[%lu, %lu, %lu]], [[%lu, %lu, %lu]]\n",
            task->pid,
            task->pgd_alloc_count, task->pgd_free_count, task->pgd_set_count,
            task->pud_alloc_count, task->pud_free_count, task->pud_set_count,
            task->pmd_alloc_count, task->pmd_free_count, task->pmd_set_count,
            task->pte_alloc_count, task->pte_free_count, task->pte_set_count);
    }
    return 0;
}


/**
 * pg_stats_open - Function to handle file opening for /proc/pg_stats
 * @inode: Pointer to the inode structure
 * @file: Pointer to the file structure
 *
 * This function is called when a user reads /proc/pg_stats.
 * It uses the single_open helper to set up sequential file reading.
 */
static int pg_stats_open(struct inode *inode, struct file *file) {
    return single_open(file, pg_stats_show, NULL);
}

/**
 * File operations structure for the /proc/pg_stats entry
 *
 * Defines the functions that will be used when reading from the file.
 */
static const struct proc_ops pg_stats_fops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

/**
 * pg_stats_init - Module initialization function
 *
 * This function creates the /proc/pg_stats entry when the module is loaded.
 * It prints a kernel log message confirming successful creation.
 */
static int __init pg_stats_init(void) {
    proc_create(PROC_FILENAME, 0, NULL, &pg_stats_fops);
    pr_info("pg_stats: Module loaded, /proc/pg_stats created.\n");
    return 0;
}

/**
 * pg_stats_exit - Module cleanup function
 *
 * This function removes the /proc/pg_stats entry when the module is unloaded.
 * It prints a kernel log message confirming removal.
 */
static void __exit pg_stats_exit(void) {
    remove_proc_entry(PROC_FILENAME, NULL);
    pr_info("pg_stats: Module unloaded, /proc/pg_stats removed.\n");
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);


