// SPDX-License-Identifier: GPL-2.0
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/sched/signal.h>

// Define the format for pg_stats output
static void pg_stats_format(struct seq_file *m, struct task_struct *task)
{
    seq_printf(m, "%d: [[%lu,%lu,%lu], [%lu,%lu,%lu], [%lu,%lu,%lu], [%lu,%lu,%lu]]\n",
        task->pid,
        // PGD counters
        task->pg_stats.pgd_alloc, task->pg_stats.pgd_free, task->pg_stats.pgd_set,
        // PUD counters
        task->pg_stats.pud_alloc, task->pg_stats.pud_free, task->pg_stats.pud_set,
        // PMD counters
        task->pg_stats.pmd_alloc, task->pg_stats.pmd_free, task->pg_stats.pmd_set,
        // PTE counters
        task->pg_stats.pte_alloc, task->pg_stats.pte_free, task->pg_stats.pte_set);
}

// Iterate through all processes and print their stats
static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;

    rcu_read_lock(); // Use RCU to safely traverse the task list
    for_each_process(task) {
        pg_stats_format(m, task);
    }
    rcu_read_unlock();

    return 0;
}

// Set up the seq_file interface
static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

// Define file operations for the proc entry
static const struct proc_ops pg_stats_fops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

// Initialize the proc entry
static int __init pg_stats_init(void)
{
    proc_create("pg_stats", 0, NULL, &pg_stats_fops);
    return 0;
}

// Cleanup the proc entry
static void __exit pg_stats_exit(void)
{
    remove_proc_entry("pg_stats", NULL);
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Your Name");
MODULE_DESCRIPTION("Page Table Action Statistics");
