#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/sched/signal.h>
#include <linux/atomic.h>

static void *pg_stats_start(struct seq_file *m, loff_t *pos) {
    struct task_struct *task;
    loff_t n = *pos;

    rcu_read_lock();

    task = &init_task;
    while (n > 0) {
        task = next_task(task);
        if (task == &init_task) return NULL;
        --n;
    }

    return task;
}

static void *pg_stats_next(struct seq_file *m, void *v, loff_t *pos) {
    struct task_struct *task = v;

    ++(*pos);
    task = next_task(task);
    if (task == &init_task) return NULL;

    return task;
}

static void pg_stats_stop(struct seq_file *m, void *v) {
    rcu_read_unlock();
}

static int pg_stats_show(struct seq_file *m, void *v) {
    struct task_struct *task = v;

    seq_printf(m, "%d: [%lu,%lu,%lu] [%lu,%lu,%lu] [%lu,%lu,%lu] [%lu,%lu,%lu]\n",

              task->pid,

              atomic_long_read(&task->pgd_alloc),
              atomic_long_read(&task->pgd_free),
              atomic_long_read(&task->pgd_set),

              atomic_long_read(&task->pud_alloc),
              atomic_long_read(&task->pud_free),
              atomic_long_read(&task->pud_set),

              atomic_long_read(&task->pmd_alloc),
              atomic_long_read(&task->pmd_free),
              atomic_long_read(&task->pmd_set),

              atomic_long_read(&task->pte_alloc),
              atomic_long_read(&task->pte_free),
              atomic_long_read(&task->pte_set));
    return 0;
}

static const struct seq_operations pg_stats_seq_ops = { // an LLM advised that I include seq_ops
    .start = pg_stats_start,
    .next  = pg_stats_next,
    .stop  = pg_stats_stop,
    .show  = pg_stats_show,
};

static int pg_stats_open(struct inode *inode, struct file *file) { return seq_open(file, &pg_stats_seq_ops); }

static const struct proc_ops pg_stats_file_ops = { // an LLM advised that I include file ops
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = seq_release,
};

static int __init pg_stats_init(void) {
    proc_create("pg_stats", 0, NULL, &pg_stats_file_ops);
    return 0;
}
fs_initcall(pg_stats_init);
